unit avidata;

interface

uses
    Windows,
    MMSystem,
    Messages,
    SysUtils,
    Classes,
    vfw;

type
  TVideoCodecs = array[1..136, 1..2] of string;
  TAudioCodecs = array[1..129, 1..2] of string;
const
  VIDEO_CODEC_COUNT: integer = 136;
  AUDIO_CODEC_COUNT: integer = 63;
  DEBUG: boolean = false;
  videocodecs: TVideoCodecs =
  (
    ('3IVX','3ivx MPEG4-based codec.'),
    ('ANIM','Intel - RDX'),
    ('AUR2','AuraVision - Aura 2 Codec - YUV 422'),
    ('AURA','AuraVision - Aura 1 Codec - YUV 411'),
    ('ADVJ','Avid M-JPEG'),
    ('AFLI','Autodesk Animator codec'),
    ('AFLC','Autodesk Animator codec'),
    ('AP41','AngelPotion Definitive'),
    ('AVRN','Avid M-JPEG'),
    ('BT20','Brooktree - MediaStream codec'),
    ('BTCV','Brooktree - Composite Video codec'),
    ('CC12','Intel - YUV12 codec'),
    ('CDVC','Canopus - DV codec'),
    ('CHAM','Winnov, Inc. - MM_WINNOV_CAVIARA_CHAMPAGNE'),
    ('CMYK','Uncompressed CMYK'),
    ('CPLA','Weitek - 4:2:0 YUV Planar'),
    ('CRAM','Microsoft - Video 1'),
    ('CVID','Supermac - Cinepak'),
    ('CWLT','reserved'),
    ('DIV3','DivX 3'),
    ('DIV4','DivX 4'),
    ('DIVX','OpenDivX'),
    ('DUCK','Duck Corp. - TrueMotion 1.0'),
    ('DV25','DVCPRO Matrox'),
    ('DV50','DVCPRO50 Matrox'),
    ('DVE2','InSoft - DVE-2 Videoconferencing codec'),
    ('DVDS','Generic DV Codec'),
    ('DXT1','reserved'),
    ('DXT2','reserved'),
    ('DXT3','reserved'),
    ('DXT4','reserved'),
    ('DXT5','reserved'),
    ('DXTC','DirectX Texture Compression'),
    ('FLJP','D-Vision - Field Encoded Motion JPEG With LSI Bitstream Format'),
    ('GWLT','reserved'),
    ('H260','Intel - Conferencing codec'),
    ('H261','Intel - Conferencing codec'),
    ('H262','Intel - Conferencing codec'),
    ('H263','Intel - Conferencing codec'),
    ('H264','Intel - Conferencing codec'),
    ('H265','Intel - Conferencing codec'),
    ('H266','Intel - Conferencing codec'),
    ('H267','Intel - Conferencing codec'),
    ('H268','Intel - Conferencing codec'),
    ('H269','Intel - Conferencing codec'),
    ('HFYU','Huffman Lossless Codec'),
    ('I263','Intel - I263'),
    ('I420','Intel - Indeo 4 codec'),
    ('IAN ','Intel - RDX'),
    ('ICLB','InSoft - CellB Videoconferencing codec'),
    ('ILVC','Intel - Layered Video'),
    ('ILVR','ITU-T - H.263+ compression standard'),
    ('IRAW','Intel - YUV uncompressed'),
    ('IV30','Intel - Indeo Video 3 codec'),
    ('IV31','Intel - Indeo Video 3.1 codec'),
    ('IV32','Intel - Indeo Video 3 codec'),
    ('IV33','Intel - Indeo Video 3 codec'),
    ('IV34','Intel - Indeo Video 3 codec'),
    ('IV35','Intel - Indeo Video 3 codec'),
    ('IV36','Intel - Indeo Video 3 codec'),
    ('IV37','Intel - Indeo Video 3 codec'),
    ('IV38','Intel - Indeo Video 3 codec'),
    ('IV39','Intel - Indeo Video 3 codec'),
    ('IV40','Intel - Indeo Video 4 codec'),
    ('IV41','Intel - Indeo Video 4 codec'),
    ('IV42','Intel - Indeo Video 4 codec'),
    ('IV43','Intel - Indeo Video 4 codec'),
    ('IV44','Intel - Indeo Video 4 codec'),
    ('IV45','Intel - Indeo Video 4 codec'),
    ('IV46','Intel - Indeo Video 4 codec'),
    ('IV47','Intel - Indeo Video 4 codec'),
    ('IV48','Intel - Indeo Video 4 codec'),
    ('IV49','Intel - Indeo Video 4 codec'),
    ('IV50','Intel - Indeo 5.0'),
    ('LEAD','LEAD Video Codec'),
    ('LJPG','LEAD MJPEG Codec'),
    ('M4S2','Microsoft - ISO Compliant MPEG-4'),
    ('MP42','Microsoft - MPEG-4 Video Codec V2'),
    ('MP43','Microsoft - MPEG-4 Video Codec V3'),
    ('MPEG','Chromatic - MPEG-1 Video I Frame'),
    ('MRCA','FAST Multimedia - Mrcodec'),
    ('MRLE','Microsoft - Run Length Encoding'),
    ('MSVC','Microsoft - Video 1'),
    ('NTN1','Nogatech - Video Compression 1'),
    ('QPEG','Q-Team - QPEG 1.1 Format video codec'),
    ('RGBT','Computer Concepts - 32 bit support'),
    ('RLE ','Microsoft Run Length Encoding'),
    ('RLE4','Microsoft 4bpp Run Length Encoding'),
    ('RLE8','Microsoft 8bpp Run Length Encoding'),
    ('RV20','RealVideo G2'),
    ('RV30','RealVideo 8'),
    ('RT21','Intel - Indeo 2.1 codec'),
    ('RVX ','Intel - RDX'),
    ('SAN3','DivX 3.11a'),
    ('SDCC','Sun Communications - Digital Camera Codec'),
    ('SFMC','Crystal Net - SFM Codec'),
    ('SMSC','Radius - proprietary'),
    ('SMSD','Radius - proprietary'),
    ('SPLC','Splash Studios - ACM audio codec'),
    ('SQZ2','Microsoft - VXtreme Video Codec V2'),
    ('SV10','Sorenson - Video R1'),
    ('TLMS','TeraLogic - Motion Intraframe Codec'),
    ('TLST','Terogic - Motion Intraframe Codec'),
    ('TM20','Duck Corp. - TrueMotion 2.0'),
    ('TMIC','TeraLogic - Motion Intraframe Codec'),
    ('TMOT','Horizons Technology - TrueMotion Video Compression Algorithm'),
    ('TR20','Duck Corp. - TrueMotion RT 2.0'),
    ('V422','Vitec Multimedia - 24 bit YUV 4:2:2 format (CCIR 601).'),
    ('V655','Vitec Multimedia - 16 bit YUV 4:2:2 format.'),
    ('VCR1','ATI - VCR 1.0'),
    ('VGPX','VideoGramPix'),
    ('VIVO','Vivo - H.263 Video Codec'),
    ('VIXL','Miro Computer Products AG'),
    ('VLV1','Videologic - VLCAP.DRV'),
    ('WBVC','Winbond Electronics - W9960'),
    ('XLV0','NetXL, Inc. - XL Video Decoder'),
    ('XVID','XviD Open Source MPEG-4'),
    ('Y411','Microsoft YUV 4:1:1'),
    ('YC12','Intel - YUV12 codec'),
    ('YUV8','Winnov, Inc. - MM_WINNOV_CAVIAR_YUV8'),
    ('YUV9','Intel - YUV9'),
    ('YUYV','Canopus - YUYV compressor'),
    ('ZPEG','Metheus - Video Zipper'),

    ('CYUV','Creative Labs, Inc - Creative Labs YUV'),
    ('DIB ','Microsoft Uncompressed'),
    ('FVF1','Iterated Systems, Inc. - Fractal Video Frame'),
    ('IF09','Intel - Intel Intermediate YUV9'),
    ('IJPG','Intergraph JPEG'),
    ('JPEG','Microsoft - Still Image JPEG DIB'),
    ('MJPG','Microsoft - Motion JPEG DIB Format'),
    ('PHMO','IBM - Photomotion'),
    ('ULTI','IBM - Ultimotion'),
    ('VDCT','Vitec Multimedia - Video Maker Pro DIB'),
    ('VIDS','Vitec Multimedia - YUV 4:2:2 CCIR 601 for V422'),
    ('YUY2','Raw, uncompressed YUV 4:2:2'),
    ('YU92','Intel - YUV')
  );
  audiocodecs: TAudioCodecs =
  (
    ('0','Unknown Wave'),                       {0x0000}
    ('1','Microsoft PCM'),                      {0x0001}
    ('2','Microsoft ADPCM'),                    {0x0002}
    ('3','IEEE Float'),                         {0x0003}
    ('4','Compaq VSELP'),                       {0x0004}
    ('5','IBM CVSD'),                           {0x0005}
    ('6','Microsoft ALAW'),                     {0x0006}
    ('7','Microsoft MULAW'),                    {0x0007}
    ('8','DTS'),                                {0x0008}
    ('16','OKI ADPCM'),                         {0x0010}
    ('17','Intel DVI ADPCM'),                   {0x0011}
    ('18','VideoLogic Mediaspace ADPCM'),       {0x0012}
    ('19','Sierra ADPCM'),                      {0x0013}
    ('20','Antex G723 ADPCM'),                  {0x0014}
    ('21','DSP Solutions DIGISTD'),             {0x0015}
    ('22','DSP Solutions DIGIFIX'),             {0x0016}
    ('23','Dialogic OKI ADPCM'), 		{0x0017}
    ('24','MediaVision ADPCM'),  		{0x0018}
    ('25','HP CU'),  				{0x0019}
    ('32','Yamaha ADPCM'),              	{0x0020}
    ('33','Speech Compression Sonarc'),         {0x0021}
    ('34','DSP Group Truespeech'),              {0x0022}
    ('35','Echo Speech EchoSC1'),               {0x0023}
    ('36','Audiofile AF36'),                    {0x0024}
    ('37','Calypso APTX'),                      {0x0025}
    ('38','AudioFile AF10'),                    {0x0026}
    ('39','Prosody 1612'),                      {0x0027}
    ('40','Merging Technologies LRC'),          {0x0028}
    ('48','Dolby AC2'),                         {0x0030}
    ('49','Microsoft GSM610'),                  {0x0031}
    ('50','Microsoft MSN Audio'),               {0x0032}
    ('51','Antex ADPCME'),                      {0x0033}
    ('52','Control Resources VQLPC'),           {0x0034}
    ('53','DSP Solutions Real'),                {0x0035}
    ('54','DSP Solutions ADPCM'),               {0x0036}
    ('55','Control Resources CR10'),            {0x0037}
    ('56','Natural MicroSystems VBXADPCM'),     {0x0038}
    ('57','Roland RDAC'),                       {0x0039}
    ('58','Echo Speech SC3'),                   {0x003A}
    ('59','Rockwell ADPCM'),                    {0x003B}
    ('60','Rockwell Digit LK'),                 {0x003C}
    ('61','Xebec Multimedia'),                  {0x003D}
    ('64','Antex Electronics G721 ADPCM'),      {0x0040}
    ('65','Antex Electronics G728 CELP'),       {0x0041}
    ('66','Microsoft G723'),           		{0x0042}
    ('80','Microsoft MPEG'),                    {0x0050}
    ('82','Voxware Metavoice'),                 {0x0052}
    ('83','PAC'),                               {0x0053}
    ('85','MPEG Layer 3'),                      {0x0055}
    ('89','Lucent G273'),                       {0x0059}
    ('96','Cirrus Logic'),                      {0x0060}
    ('97','ESS Technology PCM'),                {0x0061}
    ('98','Voxware'),                           {0x0062}
    ('99','Canopus ATRAC'),                     {0x0063}
    ('100','APICOM G726 ADPCM'),                {0x0064}
    ('101','APICOM G722 ADPCM'),                {0x0065}
    ('102','Microsoft DSAT'),                   {0x0066}
    ('103','Microsoft DSAT Display'),           {0x0067}
    ('105','Voxware Byte Aligned'),             {0x0069}
    ('112','Voxware AC8'),                      {0x0070}
    ('113','Voxware AC10'),                     {0x0071}
    ('114','Voxware AC16'),                     {0x0072}
    ('115','Voxware AC20'),                     {0x0073}
    ('116','Voxware Metavoice'),                {0x0074}
    ('117','Voxware Metasound'),                {0x0075}
    ('118','Voxware RT29HW'),                   {0x0076}
    ('119','Voxware VR12'),                     {0x0077}
    ('120','Voxware VR18'),                     {0x0078}
    ('121','Voxware TQ40'),                     {0x0079}
    ('128','Softsound'),                        {0x0080}
    ('129','Voxware TQ60'),                     {0x0081}
    ('130','Microsoft RT24'),                   {0x0082}
    ('131','AT&T G.729A'),                      {0x0083}
    ('132','MVI MV12'),                         {0x0084}
    ('133','DataFusion G.726'),                 {0x0085}
    ('134','DataFusion GSM610'),                {0x0086}
    ('136','ISIAudio'),                         {0x0088}
    ('137','OnLive'),                           {0x0089}
    ('145','Siemens SBC24'),                    {0x0091}
    ('146','Dolby AC3 SPDIF'),                  {0x0092}
    ('147','Mediasonic G273'),                  {0x0093}
    ('148','Prosody 8kbps'),                    {0x0094}
    ('151','ZyXEL ADPCM'),                      {0x0097}
    ('152','Philips LPCBB'),                    {0x0098}
    ('153','Studer Packed'),                    {0x0099}
    ('160','Malden Phonytalk'),                 {0x00A0}
    ('256','Rhetorix ADPCM'),                   {0x0100}
    ('257','BeCubed IRAT'),                     {0x0101}
    ('273','Vivo G.723'),                       {0x0111}
    ('274','Vivo Siren'),                       {0x0112}
    ('291','Digital G.723'),                    {0x0123}
    ('304','Siprolab ACEPLNET'),                {0x0130}
    ('305','Siprolab ACELP4800'),               {0x0131}
    ('306','Siprolab ACELP8V3'),                {0x0132}
    ('307','Siprolab G729'),                    {0x0133}
    ('308','Siprolab G729A'),                   {0x0134}
    ('309','Siprolab KELVIN'),                  {0x0135}
    ('320','Dictaphone G726ADPCM'),             {0x0140}
    ('336','Qualcomm Purevoice'),               {0x0150}
    ('337','Qualcomm Halfrate'),                {0x0151}
    ('341','Ring Zero TUBGSM'),                 {0x0155}
    ('352','Microsoft Windows Media Audio codec/DivX Audio'),                 {0x0160}
    ('353','Microsoft Windows Media Audio codec/DivX Audio'),                       {0x0161}
    ('512','Creative Labs ADPCM'),              {0x0200}
    ('514','Creative Labs Fastspeech 8'),       {0x0202}
    ('515','Creative Labs Fastspeech 10'),      {0x0203}
    ('528','UHER ADPCM'),                       {0x0210}
    ('544','Quarterdeck'),                      {0x0220}
    ('560','I-Link VC'),                        {0x0230}
    ('576','Aureal RAW-SPORT'),                 {0x0240}
    ('592','IPI HSX'),                          {0x0250}
    ('593','IPI RPELP'),                        {0x0251}
    ('608','Consistent CS2'),                   {0x0260}
    ('624','Sony SCX'),                         {0x0270}
    ('768','Fujitsu FM Towns Snd'),             {0x0300}
    ('1024','Brooktree TV Digital'),            {0x0400}
    ('1104','QDesign Music'),                   {0x0450}
    ('1664','AT&T VME VMPCM'),                  {0x0680}
    ('1665','AT&T TPC'),                        {0x0681}
    ('4096','Olivetti GSM'),                    {0x1000}
    ('4097','Olivetti ADPCM'),                  {0x1001}
    ('4098','Olivetti CELP'),                   {0x1002}
    ('4099','Olivetti SBC'),                    {0x1003}
    ('4100','Olivetti OPR'),                    {0x1004}
    ('4352','Lernout & Hauspie Codec'),         {0x1100}
    ('5120','Norris'),                          {0x1400}
    ('5121','ISIAudio'),                        {0x1401}
    ('5376','Soundspace Music Compression'),    {0x1500}
    ('8192','FAST Multimedia DVM')              {0x2000}

  );
type
  TAVIInfo = class
  public
    streamtype:string;
    problem:string;
    constructor Create;
    procedure setInfoDetails(pfile:PAVIFILE;fileinfo:TAVIFILEINFO;pstream:PAVISTREAM;streaminfo:TAVISTREAMINFO);virtual;abstract;
    function info:string;virtual;abstract;
  end;

  TAVIVideoInfo = class(TAVIInfo)
  private
    //video variables
    F_fps:real;
    F_Width:integer;
    F_Height:integer;
    F_Bit_Depth:integer;
    F_Frame_Count:integer;
    F_fourcc:cardinal;
    F_fourcc_string:string;
    F_Codec_Name:string;
    //video functions
  public
    constructor Create;
    procedure setInfoDetails(pfile:PAVIFILE;fileinfo:TAVIFILEINFO;pstream:PAVISTREAM;streaminfo:TAVISTREAMINFO);override;
    function info:string;override;

    function getFrameAspectRatioString():string;
    function getDurationString():string;
    function getFPSString():string;
  end;

  TAVIAudioInfo = class(TAVIInfo)
  private
    F_fourcc:cardinal;
    F_fourcc_string:string;
    F_Codec_Name:string;
    F_Sampling_Rate:cardinal;
    F_Bit_Depth:cardinal;
    F_Channels:cardinal;
  public
    constructor Create;
    procedure setInfoDetails(pfile:PAVIFILE;fileinfo:TAVIFILEINFO;pstream:PAVISTREAM;streaminfo:TAVISTREAMINFO);override;
    function info:string;override;
  end;

  TAVIMidiInfo = class(TAVIInfo)
    constructor Create;
    procedure setInfoDetails(pfile:PAVIFILE;fileinfo:TAVIFILEINFO;pstream:PAVISTREAM;streaminfo:TAVISTREAMINFO);override;
    function info:string;override;
  end;

  TAVITextInfo = class(TAVIInfo)
    constructor Create;
    procedure setInfoDetails(pfile:PAVIFILE;fileinfo:TAVIFILEINFO;pstream:PAVISTREAM;streaminfo:TAVISTREAMINFO);override;
    function info:string;override;
  end;

  TAVIData = class
  private
    _list:TList;
    _depth : integer;
    _logger:TStrings;

    function longToString(val:longint):string;
    procedure log(output:string);
  public
    constructor Create;
    procedure process(filename:string);
    property streamlist:Tlist read _list;
  end;


implementation

//Base

constructor TAVIInfo.Create;
begin
  inherited;
  problem:='';
end;

//Video

constructor TAVIVideoInfo.Create;
begin
  inherited;
  streamtype:='Video';
end;

procedure TAVIVideoInfo.setInfoDetails(pfile:PAVIFILE;fileinfo:TAVIFILEINFO;pstream:PAVISTREAM;streaminfo:TAVISTREAMINFO);
var
  i : integer;
  lSize:LongInt;
  videoarray: array of integer;
  pvideoformat:PBITMAPINFoHEADER;
  hicd: HIC;
  icinfo:TICINFO;
  test:string;
begin
  F_Codec_Name := '';

  if(AVIStreamReadFormat(pstream, AVIStreamStart(pstream), nil, @lSize)<>AVIERR_OK) then
  begin
    problem:='Video Stream Format problem';
    exit;
  end;

  setLength(videoarray,lSize);
  if(AVIStreamReadFormat(pstream, AVIStreamStart(pstream), videoarray, @lSize)<AVIERR_OK) then
  begin
    problem:='Video Stream Format problem pt2';
    exit;
  end;

  pvideoformat:=PBITMAPINFOHEADER(videoarray);

  F_fourcc := streaminfo.fccHandler;
  if (F_fourcc=0) then
    F_fourcc:=pvideoformat.biCompression;

  hicd := ICOpen(ICTYPE_VIDEO, F_fourcc, ICMODE_DECOMPRESS);
  if (hicd<>0) then
  begin
    ICGetInfo(hicd, @icinfo, sizeof(icinfo));
    F_Codec_Name:=icinfo.szDescription;
    ICClose(hicd);
  end;

  F_fourCC_string:=char(F_fourcc)+char(F_fourcc shr 8)+char(F_fourcc shr 16)+char(F_fourcc shr 24);

  if (F_Codec_Name='') then
  begin
    F_Codec_Name:='Unrecognized';
    test:=uppercase(F_Fourcc_string);
    for i := 1 to VIDEO_CODEC_COUNT do begin
      if (uppercase(videocodecs[i][1]) = test) then
      begin
        F_Codec_Name := videocodecs[i][2];
     end;
    end;
  end;

  if (F_Codec_Name='Unrecognized') then
    if ((F_fourCC_string[1]=char(0)) or (F_fourCC_string[2]=char(0)) or (F_fourCC_string[3]=char(0)) or (F_fourCC_string[4]=char(0))) then
      Str(F_Fourcc,F_fourCC_string);

  F_Width       := fileinfo.dwWidth;
  F_Height      := fileinfo.dwHeight;
  F_Bit_Depth   := pvideoformat.biBitCount;
  F_fps         := fileinfo.dwRate/fileinfo.dwScale;
  F_Frame_Count := fileinfo.dwLength;
end;

function TAVIVideoInfo.info:string;
var
  temp:string;
begin
  if (problem <> '') then begin
    result := '    Video Stream'+#13+#10+
              'Exception: '+problem;
    exit;
  end;
  if (F_Fourcc=0) then
    F_Fourcc_string:='bad avi';
  str(F_fourcc,temp);
  result := '    Video Stream'+#13+#10+
            'Codec:               '+F_Codec_Name+' ('+F_fourcc_string+')'+#13+#10;
  str(F_Width, temp);
  result := result+'Frame Size:       '+temp+' x ';
  str(F_Height, temp);
  result := result+temp+' pixels'+#13+#10;
  str(F_Bit_Depth, temp);
  result := result+'Bit Depth:          '+temp+#13+#10;
  result := result+'Aspect Ratio:    '+getFrameAspectRatioString+#13+#10;
  str(F_Frame_Count, temp);
  result := result+'Frames:             '+temp+#13+#10;
  result := result+'Frame Rate:      '+getFPSString+' fps'+#13+#10;
  result := result+'Duration:           '+getDurationString;
end;

function TAVIVideoInfo.getDurationString:string;
var
  hours, minutes, seconds:integer;
  temp:string;
begin
  seconds:=trunc(F_Frame_Count/F_fps);
  hours:=trunc(seconds/3600);
  minutes:=trunc((seconds-(hours*3600))/60);
  seconds:=seconds-(hours*3600)-(minutes*60);
  result:='';
  Str(hours,temp);
  if length(temp)<2 then
    temp:='0'+temp;
  result:=temp+':';
  Str(minutes,temp);
  if length(temp)<2 then
    temp:='0'+temp;
  result:=result+temp+':';
  Str(seconds,temp);
  if length(temp)<2 then
    temp:='0'+temp;
  result:=result+temp;
end;

function TAVIVideoInfo.getFrameAspectRatioString():string;
var
  temp:string;
  value:real;
begin
  value:=(round((F_Width/F_Height)*100000))/100000;
  if (abs(value-1.78))<0.00001 then
    temp := '16:9'
  else if (abs(value-1.33333))<0.00001 then
    temp := '4:3'
  else if (abs(value-1.25))<0.00001 then
    temp := '5:4'
  else if (abs(value-1.5))<0.00001 then
    temp := '3:2'
  else if (abs(value-1.0))<0.00001 then
    temp := '1:1'
  else if (abs(value-1.81818))<0.00001 then
    temp := '20:11'
  else begin
    Str(value:4:2,temp);
    temp:=temp+':1';
  end;
  result := temp;
end;

function TAVIVideoInfo.getFPSString:string;
var
  fps:integer;
begin
  if (abs(F_fps-29.97)<0.001) then
    result:='29.97'
  else if (abs(F_fps-23.976)<0.001) then
    result:='23.976'
  else begin
    fps:= round(F_fps);
    Str(fps, result);
  end;
end;

//AUDIO

constructor TAVIAudioInfo.Create;
begin
  inherited;
  streamtype:='Audio';
end;

procedure TAVIAudioInfo.setInfoDetails(pfile:PAVIFILE;fileinfo:TAVIFILEINFO;pstream:PAVISTREAM;streaminfo:TAVISTREAMINFO);
var
  i : integer;
  lSize:LongInt;
  wavearray: array of integer;
  pwaveformat:PWAVEFORMATEX;
begin
  if(AVIStreamReadFormat(pstream, AVIStreamStart(pstream), nil, @lSize)<>AVIERR_OK) then
  begin
    problem:='Audio Stream Format problem pt1';
    exit;
  end;

  setLength(wavearray,lSize);
  if(AVIStreamReadFormat(pstream, AVIStreamStart(pstream), wavearray, @lSize)<AVIERR_OK) then
  begin
    problem:='Audio Stream Format problem pt2';
    exit;
  end;

  pwaveformat:=PWAVEFORMATEX(wavearray);

  F_fourcc:=pwaveformat.wFormatTag;
  Str(F_fourcc, F_fourcc_string);
  F_Sampling_Rate:=pwaveformat.nSamplesPerSec;
  F_Bit_Depth:=pwaveformat.wBitsPerSample;
  F_Channels:=pwaveformat.nChannels;

  F_Codec_Name := 'Unrecognized';
  for i := 1 to AUDIO_CODEC_COUNT do begin
    if (audiocodecs[i][1] = F_fourCC_string) then
    begin
      F_Codec_Name := audiocodecs[i][2];
    end;
  end;

end;

function TAVIAudioInfo.info:string;
var
  temp:string;
begin
  if (problem <> '') then begin
    result := '    Audio Stream'+#13+#10+
              'Exception: '+problem;
    exit;
  end;
  result:='    Audio Stream'+#13+#10+
          'Codec:               '+F_Codec_Name+' ('+F_fourcc_string+')'+#13+#10;
  str(F_Channels,temp);
  if (F_Channels=1) then
    temp:=temp+' (Mono)'
  else if (F_Channels=2) then
    temp:=temp+' (Stereo)';
  result:=result+'Channels:          '+temp+#13+#10;
  if (F_Bit_Depth=0) then
    result:=result+'Bit Depth:          '+'n/a'+#13+#10
  else
  begin
    str(F_Bit_Depth,temp);
    result:=result+'Bit Depth:          '+temp+#13+#10;
  end;
  str(F_Sampling_Rate,temp);
  result:=result+'Sampling Rate:  '+temp + ' Hz';

//             'Codec:               '+   avi.AudioCodecNameAsString+' ('+avi.AudioFourCCAsString+')'#10+#13+
//             'Channels:          '+   avi.AudioChannelCountAsString+#10+#13+
//             'Bit Depth:          '+   avi.AudioBitDepthAsString+#10+#13+
//             'Sampling Rate:  '+   avi.AudioSamplingRateAsString;
end;

//MIDI

constructor TAVIMidiInfo.Create;
begin
  inherited;
  streamtype:='Midi';
end;

procedure TAVIMidiInfo.setInfoDetails(pfile:PAVIFILE;fileinfo:TAVIFILEINFO;pstream:PAVISTREAM;streaminfo:TAVISTREAMINFO);
begin
end;

function TAVIMidiInfo.info:string;
begin
  result:='    Midi Stream'+#13+#10+
          'Not supported'+#13+#10;
end;

//TEXT

constructor TAVITextInfo.Create;
begin
  inherited;
  streamtype:='Text';
end;

procedure TAVITextInfo.setInfoDetails(pfile:PAVIFILE;fileinfo:TAVIFILEINFO;pstream:PAVISTREAM;streaminfo:TAVISTREAMINFO);
begin
end;

function TAVITextInfo.info:string;
begin
  result:='    Text Stream'+#13+#10+
          'Not supported'+#13+#10;
end;

//AVIDATA

constructor TAVIData.Create;
begin
  inherited;
  _list := TList.Create;
  _logger:=nil;
end;

procedure TAVIData.process(filename:string);
var
  avifile:  PAVIFILE;
  avifile_info: TAVIFILEINFO;
  avistream:  PAVISTREAM ;
  avistream_info: TAVISTREAMINFO ;
  i:integer;
  info:TAVIInfo;
begin
  AVIFileInit();
  try
    if(AVIFileOpen(avifile,PChar(filename),OF_READ,nil)<>AVIERR_OK) then
      raise Exception.Create('Couldn''t open file '+filename+' - does not exist or is not valid avi');
    AVIFileInfo(avifile,@avifile_info,sizeof(avifile_info));
    for i:=0 to 100 do begin
      try
        if(AVIFileGetStream(avifile,avistream,0,i)<>AVIERR_OK) then
          break;
        AVIStreamInfo(avistream,@avistream_info,sizeof(avistream_info));
        case avistream_info.fccType of
          streamtypeVIDEO:
            begin
              info:=TAVIVideoInfo.Create;
              _list.Add(info);
              info.setInfoDetails(avifile,avifile_info,avistream,avistream_info);
            end;
          streamtypeAUDIO:
            begin
              info:=TAVIAudioInfo.Create;
              _list.Add(info);
              info.setInfoDetails(avifile,avifile_info,avistream,avistream_info);
            end;
          streamtypeMIDI:
            begin
              info:=TAVIMidiInfo.Create;
              _list.Add(info);
              info.setInfoDetails(avifile,avifile_info,avistream,avistream_info);
            end;
          streamtypeTEXT:
            begin
              info:=TAVITextInfo.Create;
              _list.Add(info);
              info.setInfoDetails(avifile,avifile_info,avistream,avistream_info);
            end;
          else
            begin
            end;
        end;
      finally
        if (avistream<>nil) then
          AVIStreamRelease(avistream);
      end;
    end;
  finally
    if (avifile<>nil) then begin
      AVIFileRelease(avifile);
    end;
    AVIFileExit();
  end;
end;

function TAVIData.longToString(val:longint):string;
var
  temp:string;
begin
  Str(val, temp);
  result := temp;
end;

procedure TAVIData.log(output:string);
var
  temp:string;
  i:integer;
begin
  if (DEBUG = true) then begin
    temp:='';
    for i:=1 to _depth-2 do
      temp:=temp+' ';
    if (_logger=nil) then
      writeln(temp+output)
    else
      _logger.Add(temp+output);
  end;
end;

end.

{/* WAVE form wFormatTag IDs */
#define  WAVE_FORMAT_UNKNOWN    0x0000  /*  Microsoft Corporation  */
#define  WAVE_FORMAT_ADPCM      0x0002  /*  Microsoft Corporation  */
#define  WAVE_FORMAT_IEEE_FLOAT 0x0003  /*  Microsoft Corporation  */
                                        /*  IEEE754: range (+1, -1]  */
                                        /*  32-bit/64-bit format as defined by */
                                        /*  MSVC++ float/double type */
#define  WAVE_FORMAT_IBM_CVSD   0x0005  /*  IBM Corporation  */
#define  WAVE_FORMAT_ALAW       0x0006  /*  Microsoft Corporation  */
#define  WAVE_FORMAT_MULAW      0x0007  /*  Microsoft Corporation  */
#define  WAVE_FORMAT_OKI_ADPCM  0x0010  /*  OKI  */
#define  WAVE_FORMAT_DVI_ADPCM  0x0011  /*  Intel Corporation  */
#define  WAVE_FORMAT_IMA_ADPCM  (WAVE_FORMAT_DVI_ADPCM) /*  Intel Corporation  */
#define  WAVE_FORMAT_MEDIASPACE_ADPCM   0x0012  /*  Videologic  */
#define  WAVE_FORMAT_SIERRA_ADPCM       0x0013  /*  Sierra Semiconductor Corp  */
#define  WAVE_FORMAT_G723_ADPCM 0x0014  /*  Antex Electronics Corporation  */
#define  WAVE_FORMAT_DIGISTD    0x0015  /*  DSP Solutions, Inc.  */
#define  WAVE_FORMAT_DIGIFIX    0x0016  /*  DSP Solutions, Inc.  */
#define  WAVE_FORMAT_DIALOGIC_OKI_ADPCM 0x0017  /*  Dialogic Corporation  */
#define  WAVE_FORMAT_MEDIAVISION_ADPCM  0x0018  /*  Media Vision, Inc. */
#define  WAVE_FORMAT_YAMAHA_ADPCM       0x0020  /*  Yamaha Corporation of America  */
#define  WAVE_FORMAT_SONARC     0x0021  /*  Speech Compression  */
#define  WAVE_FORMAT_DSPGROUP_TRUESPEECH        0x0022  /*  DSP Group, Inc  */
#define  WAVE_FORMAT_ECHOSC1    0x0023  /*  Echo Speech Corporation  */
#define  WAVE_FORMAT_AUDIOFILE_AF36     0x0024  /*    */
#define  WAVE_FORMAT_APTX       0x0025  /*  Audio Processing Technology  */
#define  WAVE_FORMAT_AUDIOFILE_AF10     0x0026  /*    */
#define  WAVE_FORMAT_DOLBY_AC2  0x0030  /*  Dolby Laboratories  */
#define  WAVE_FORMAT_GSM610     0x0031  /*  Microsoft Corporation  */
#define  WAVE_FORMAT_MSNAUDIO   0x0032  /*  Microsoft Corporation  */
#define  WAVE_FORMAT_ANTEX_ADPCME       0x0033  /*  Antex Electronics Corporation  */
#define  WAVE_FORMAT_CONTROL_RES_VQLPC  0x0034  /*  Control Resources Limited  */
#define  WAVE_FORMAT_DIGIREAL   0x0035  /*  DSP Solutions, Inc.  */
#define  WAVE_FORMAT_DIGIADPCM  0x0036  /*  DSP Solutions, Inc.  */
#define  WAVE_FORMAT_CONTROL_RES_CR10   0x0037  /*  Control Resources Limited  */
#define  WAVE_FORMAT_NMS_VBXADPCM       0x0038  /*  Natural MicroSystems  */
#define  WAVE_FORMAT_CS_IMAADPCM 0x0039 /* Crystal Semiconductor IMA ADPCM */
#define  WAVE_FORMAT_ECHOSC3     0x003A /* Echo Speech Corporation */
#define  WAVE_FORMAT_ROCKWELL_ADPCM     0x003B  /* Rockwell International */
#define  WAVE_FORMAT_ROCKWELL_DIGITALK  0x003C  /* Rockwell International */
#define  WAVE_FORMAT_XEBEC      0x003D  /* Xebec Multimedia Solutions Limited */
#define  WAVE_FORMAT_G721_ADPCM 0x0040  /*  Antex Electronics Corporation  */
#define  WAVE_FORMAT_G728_CELP  0x0041  /*  Antex Electronics Corporation  */
#define  WAVE_FORMAT_MPEG       0x0050  /*  Microsoft Corporation  */
#define  WAVE_FORMAT_MPEGLAYER3 0x0055  /*  ISO/MPEG Layer3 Format Tag */
#define  WAVE_FORMAT_CIRRUS     0x0060  /*  Cirrus Logic  */
#define  WAVE_FORMAT_ESPCM      0x0061  /*  ESS Technology  */
#define  WAVE_FORMAT_VOXWARE    0x0062  /*  Voxware Inc  */
#define  WAVEFORMAT_CANOPUS_ATRAC       0x0063  /*  Canopus, co., Ltd.  */
#define  WAVE_FORMAT_G726_ADPCM 0x0064  /*  APICOM  */
#define  WAVE_FORMAT_G722_ADPCM 0x0065  /*  APICOM      */
#define  WAVE_FORMAT_DSAT       0x0066  /*  Microsoft Corporation  */
#define  WAVE_FORMAT_DSAT_DISPLAY       0x0067  /*  Microsoft Corporation  */
#define  WAVE_FORMAT_SOFTSOUND  0x0080  /*  Softsound, Ltd.      */
#define  WAVE_FORMAT_RHETOREX_ADPCM     0x0100  /*  Rhetorex Inc  */
#define  WAVE_FORMAT_CREATIVE_ADPCM     0x0200  /*  Creative Labs, Inc  */
#define  WAVE_FORMAT_CREATIVE_FASTSPEECH8       0x0202  /*  Creative Labs, Inc  */
#define  WAVE_FORMAT_CREATIVE_FASTSPEECH10      0x0203  /*  Creative Labs, Inc  */
#define  WAVE_FORMAT_QUARTERDECK 0x0220 /*  Quarterdeck Corporation  */
#define  WAVE_FORMAT_FM_TOWNS_SND       0x0300  /*  Fujitsu Corp.  */
#define  WAVE_FORMAT_BTV_DIGITAL        0x0400  /*  Brooktree Corporation  */
#define  WAVE_FORMAT_OLIGSM     0x1000  /*  Ing C. Olivetti & C., S.p.A.  */
#define  WAVE_FORMAT_OLIADPCM   0x1001  /*  Ing C. Olivetti & C., S.p.A.  */
#define  WAVE_FORMAT_OLICELP    0x1002  /*  Ing C. Olivetti & C., S.p.A.  */
#define  WAVE_FORMAT_OLISBC     0x1003  /*  Ing C. Olivetti & C., S.p.A.  */
#define  WAVE_FORMAT_OLIOPR     0x1004  /*  Ing C. Olivetti & C., S.p.A.  */
#define  WAVE_FORMAT_LH_CODEC   0x1100  /*  Lernout & Hauspie  */
#define  WAVE_FORMAT_NORRIS     0x1400  /*  Norris Communications, Inc.  */
}

{.1 Microsoft Unknown Wave Format
WAVE form Registration Number (hex): 0x0000 Codec ID in the IANA Namespace: audio/vnd.wave;codec=0 WAVE form wFormatTag ID: WAVE_FORMAT_UNKNOWN Contact: Terri Hendry, 425-936-2069 Microsoft Corporation One Microsoft Way Redmond, WA 98052-6399 USA

A.2 Microsoft PCM Format
WAVE form Registration Number (hex): 0x0001 Codec ID in the IANA Namespace: audio/vnd.wave;codec=1 WAVE form wFormatTag ID: WAVE_FORMAT_PCM Contact: Terri Hendry 425-936-2069 Microsoft Corporation One Microsoft Way Redmond, WA 98052-6399 USA

A.3 Microsoft ADPCM Format
WAVE form Registration Number (hex): 0x0002 Codec ID in the IANA Namespace: audio/vnd.wave;codec=2 WAVE form wFormatTag ID: WAVE_FORMAT_ADPCM Contact: Terri Hendry 425-936-2069 Microsoft Corporation One Microsoft Way Redmond, WA 98052-6399 USA

A.4 IEEE Float
WAVE form Registration Number (hex): 0x0003 Codec ID in the IANA Namespace: audio/vnd.wave;codec=3 WAVE form wFormatTag ID: WAVE_FORMAT_IEEE_FLOAT Contact: Terri Hendry 425-936-2069 Microsoft Corporation One Microsoft Way Redmond, WA 98052-6399 USA

A.5 Compaq Computer's VSELP
WAVE form Registration Number (hex): 0x0004 Codec ID in the IANA Namespace: audio/vnd.wave;codec=4 WAVE form wFormatTag ID: WAVE_FORMAT_VSELP Additional information: VSELP codec for Windows CE 2.0 devices Contact: Doug Stewart 713-374-7925 Compaq Computer Corporation 20555 SH 249 Houston, TX 77269-2000 USA

A.6 IBM CVSD
WAVE form Registration Number (hex): 0x0005 Codec ID in the IANA Namespace: audio/vnd.wave;codec=5 WAVE form wFormatTag ID: WAVE_FORMAT_IBM_CVSD Contact: IBM Corporation

A.7 Microsoft ALAW
WAVE form Registration Number (hex): 0x0006 Codec ID in the IANA Namespace: audio/vnd.wave;codec=6 WAVE form wFormatTag ID: WAVE_FORMAT_ALAW Contact: Terri Hendry 425-936-2069 Microsoft Corporation One Microsoft Way Redmond, WA 98052-6399 USA

A.8 Microsoft MULAW
WAVE form Registration Number (hex): 0x0007 Codec ID in the IANA Namespace: audio/vnd.wave;codec=7 WAVE form wFormatTag ID: WAVE_FORMAT_MULAW Contact: Terri Hendry 425-936-2069 Microsoft Corporation One Microsoft Way Redmond, WA 98052-6399 USA

A.9 OKI ADPCM
WAVE form Registration Number (hex): 0x0010 Codec ID in the IANA Namespace: audio/vnd.wave;codec=10 WAVE form wFormatTag ID: WAVE_FORMAT_OKI_ADPCM Contact: Oki

A.10 Intel's DVI ADPCM
WAVE form Registration Number (hex): 0x0011 Codec ID in the IANA Namespace: audio/vnd.wave;codec=11 WAVE form wFormatTag ID: WAVE_FORMAT_DVI_ADPCM Contact: Intel Corporation 5200 NE Elam Young Parkway Hillsboro, OR 97124 503-696-2448

A.11 Videologic's MediaSpace ADPCM
WAVE form Registration Number (hex): 0x0012 Codec ID in the IANA Namespace: audio/vnd.wave;codec=12 WAVE form wFormatTag ID: WAVE_FORMAT_MEDIASPACE_ADPCM Contact: Videologic Home Park Estate Kings Langley England WD4 8LZ Telephone: 44-92-326-0511

A.12 Sierra ADPCM
WAVE form Registration Number (hex): 0x0013 Codec ID in the IANA Namespace: audio/vnd.wave;codec=13 WAVE form wFormatTag ID: WAVE_FORMAT_SIERRA_ADPCM Contact: Stuart Goldstein 72170.301@compuserve.com Sierra Semiconductor Corp 2075 North Capitol Avenue San Jose, California 95132 USA 408-263-9300

A.13 G.723 ADPCM
WAVE form Registration Number (hex): 0x0014 Codec ID in the IANA Namespace: audio/vnd.wave;codec=14 WAVE form wFormatTag ID: WAVE_FORMAT_G723_ADPCM Contact: Bob Bauman 310-532-3092 Antex Electronics Coporation 3184-H Airway Ave. Costa Mesa, California 92627 USA

A.14 DSP Solution's DIGISTD
WAVE form Registration Number (hex): 0x0015 Codec ID in the IANA Namespace: audio/vnd.wave;codec=15 WAVE form wFormatTag ID: WAVE_FORMAT_DIGISTD Contact: DSP Solutions, Inc 2464 Embarcadero Way Palo Alto, California 94303 USA 415-494-8086

A.15 DSP Solution's DIGIFIX
WAVE form Registration Number (hex): 0x0016 Codec ID in the IANA Namespace: audio/vnd.wave;codec=16 WAVE form wFormatTag ID: WAVE_FORMAT_DIGIFIX Contact: DSP Solutions, Inc 2464 Embarcadero Way Palo Alto, California 94303 USA 415-494-8086

A.16 Dialogic OKI ADPCM
WAVE form Registration Number (hex): 0x0017 Codec ID in the IANA Namespace: audio/vnd.wave;codec=17 WAVE form wFormatTag ID: WAVE_FORMAT_DIALOGIC_OKI_ADPCM WAVEFORMAT use: for OKI ADPCM chips or firmware Contact: Dialogic Corporation 300 Littleton Road Parsippany, NJ 07054 USA 201-334-1268

A.17 MediaVision ADPCM
WAVE form Registration Number (hex): 0x0018 Codec ID in the IANA Namespace: audio/vnd.wave;codec=18 WAVE form wFormatTag ID: WAVE_FORMAT_MEDIAVISION_ADPCM WAVEFORMAT Name: ADPCM for Jazz 16 chip set Contact: Alex Cheng Media Vision, Inc California USA

A.18 HP CU
WAVE form Registration Number (hex): 0x0019 Codec ID in the IANA Namespace: audio/vnd.wave;codec=19 WAVE form wFormatTag ID: WAVE_FORMAT_CU_CODEC Contact: Cliff Chiang Telephone: 65-3747005 Hewlett-Packard Company 452 Alexandra Road Singapore 119961 Singapore

A.19 Yamaha ADPCM
WAVE form Registration Number (hex): 0x0020 Codec ID in the IANA Namespace: audio/vnd.wave;codec=20 WAVE form wFormatTag ID: WAVE_FORMAT_YAMAHA_ADPCM Contact: Yamaha Corporation of America Systems Technology Division 981 Ridder Park Drive San Jose, California 95131 USA 408-437-3133

A.20 Speech Compression's Sonarc
WAVE form Registration Number (hex): 0x0021 Codec ID in the IANA Namespace: audio/vnd.wave;codec=21 WAVE form wFormatTag ID: WAVE_FORMAT_SONARC Contact: Speech Compression

A.21 DSP Group's True Speech
WAVE form Registration Number (hex): 0x0022 Codec ID in the IANA Namespace: audio/vnd.wave;codec=22 WAVE form wFormatTag ID: WAVE_FORMAT_DSPGROUP_TRUESPEECH Contact: DSP Group, Inc 2464 Embarcadero Way Palo Alto, California 94303 USA 415-494-8086

A.22 Echo Speech's EchoSC1
WAVE form Registration Number (hex): 0x0023 Codec ID in the IANA Namespace: audio/vnd.wave;codec=23 WAVE form wFormatTag ID: WAVE_FORMAT_ECHOSC1 Contact: Billy Brackenridge billy@isi.edu Echo Speech Corporation 6460 Via Real Carpinteria, California 93013 USA 805-684-4593

A.23 Audiofile AF36
WAVE form Registration Number (hex): 0x0024 Codec ID in the IANA Namespace: audio/vnd.wave;codec=24 WAVE form wFormatTag ID: WAVE_FORMAT_AUDIOFILE_AF36 Contact: Alan Miller 617-271-0900 Virtual Music, Inc. 19 Crosby Drive Bedford, MA 01730-1419 USA

A.24 APTX
WAVE form Registration Number (hex): 0x0025 Codec ID in the IANA Namespace: audio/vnd.wave;codec=25 WAVE form wFormatTag ID: WAVE_FORMAT_APTX Contact: Calypso Software Audio Processing Technology Edgewater Road Belfast, Northern Ireland 44-232-371110

A.25 AudioFile AF10
WAVE form Registration Number (hex): 0x0026 Codec ID in the IANA Namespace: audio/vnd.wave;codec=26 WAVE form wFormatTag ID: WAVE_FORMAT_AUDIOFILE_AF10 Contact: Alan Miller 617-271-0900 Virtual Music, Inc. 19 Crosby Drive Bedford, MA 01730-1419 USA

A.26 Prosody 1612
WAVE form Registration Number (hex): 0x0027 Codec ID in the IANA Namespace: audio/vnd.wave;codec=27 WAVE form wFormatTag ID: WAVE_FORMAT_PROSODY_1612 Additional Information: Prosody CTI Speech Card Contact: Phil Cambridge Phil.Cambridge@aculab.com Aculab plc Lakeside, Bramley Road Mount Farm, Milton Keynes MK1 1PT UK +44 1908 273800

A.27 LRC
WAVE form Registration Number (hex): 0x0028 Codec ID in the IANA Namespace: audio/vnd.wave;codec=28 WAVE form wFormatTag ID: WAVE_FORMAT_LRC Contact: Patrick Wassmer pwassmer@merging.com +41 21 931 50 11 Merging Technologies S.A. Le Verney, E Puidoux, Switzerland CH-1604

A.28 Dolby AC2
WAVE form Registration Number (hex): 0x0030 Codec ID in the IANA Namespace: audio/vnd.wave;codec=30 WAVE form wFormatTag ID: WAVE_FORMAT_DOLBY_AC2 Contact: Dolby Laboratories 100 Portrero Avenue San Francisco, California 94103-4813 USA 415-558-0200

A.29 GSM610
WAVE form Registration Number (hex): 0x0031 Codec ID in the IANA Namespace: audio/vnd.wave;codec=31 WAVE form wFormatTag ID: WAVE_FORMAT_GSM610 Contact: Terri Hendry, 425-936-2069 Microsoft Corporation One Microsoft Way Redmond, WA 98052-6399 USA 

A.30 MSNAudio 
WAVE form Registration Number (hex): 0x0032 Codec ID in the IANA Namespace: audio/vnd.wave;codec=32 WAVE form wFormatTag ID: WAVE_FORMAT_MSNAUDIO WAVEFORMAT Name: Microsoft MSN Audio Codec Contact: Terri Hendry, 425-936-2069 Microsoft Corporation One Microsoft Way Redmond, WA 98052-6399 USA 

A.31 Antex ADPCME 
WAVE form Registration Number (hex): 0x0033 Codec ID in the IANA Namespace: audio/vnd.wave;codec=33 WAVE form wFormatTag ID: WAVE_FORMAT_ANTEX_ADPCME Contact: Bob Bauman Antex Electronics Corporation 3184-H Airway Ave. Costa Mesa, California 92627 USA 310-532-3092 

A.32 Control Res VQLPC 
WAVE form Registration Number (hex): 0x0034 Codec ID in the IANA Namespace: audio/vnd.wave;codec=34 WAVE form wFormatTag ID: WAVE_FORMAT_CONTROL_RES_VQLPC Contact: Charles Larson Control Resources Limited PO Box 8694 Roland Heights, California 91748 USA 818-912-5722 

A.33 Digireal 
WAVE form Registration Number (hex): 0x0035 Codec ID in the IANA Namespace: audio/vnd.wave;codec=35 WAVE form wFormatTag ID: WAVE_FORMAT_DIGIREAL Contact: DSP Solutions, Inc 2464 Embarcadero Way Palo Alto, California 94303 USA 415-494-8086 

A.34 DigiADPCM 
WAVE form Registration Number (hex): 0x0036 Codec ID in the IANA Namespace: audio/vnd.wave;codec=36 WAVE form wFormatTag ID: WAVE_FORMAT_DIGIADPCM Contact: DSP Solutions, Inc 2464 Embarcadero Way Palo Alto, California 94303 USA 415-494-8086 

A.35 Control Res CR10 
WAVE form Registration Number (hex): 0x0037 Codec ID in the IANA Namespace: audio/vnd.wave;codec=37 WAVE form wFormatTag ID: WAVE_FORMAT_CONTROL_RES_CR10 Contact: Charles Larson Control Resources Limited PO Box 8694 Roland Heights, California 91748 USA 818-912-5722 

A.36 NMS VBXADPCM 
WAVE form Registration Number (hex): 0x0038 Codec ID in the IANA Namespace: audio/vnd.wave;codec=38 WAVE form wFormatTag ID: WAVE_FORMAT_NMS_VBXADPCM Contact: Joel Feldman, Steve Mors Natural MicroSystems 

A.37 Roland RDAC 
WAVE form Registration Number (hex): 0x0039 Codec ID in the IANA Namespace: audio/vnd.wave;codec=39 WAVE form wFormatTag ID: WAVE_FORMAT_ROLAND_RDAC WAVEFORMAT Name: Roland RDAC Proprietary format Contact: Takera Tanigawa email: tanigawa@roland.co.jp 001-81-6-682-4584 

A.38 EchoSC3 
WAVE form Registration Number (hex): 0x003A Codec ID in the IANA Namespace: audio/vnd.wave;codec=3A WAVE form wFormatTag ID: WAVE_FORMAT_ECHOSC3 WAVEFORMAT Description: Proprietary compressed format Contact: Billy Brackenridge billy@isi.edu Echo Speech Corporation 6460 Via Real Carpinteria, California 93013 USA 805-684-4593 

A.39 Rockwell ADPCM 
WAVE form Registration Number (hex): 0x003B Codec ID in the IANA Namespace: audio/vnd.wave;codec=3B WAVE form wFormatTag ID: WAVE_FORMAT_ROCKWELL_ADPCM WAVEFORMAT Name: Rockwell ADPCM Contact: Rockwell International Digital Communications Division 4311 Jamboree Rd. PO Box C Newport Beach, California 92658-8902 USA 714-833-4600 

A.40 Rockwell Digit LK 
WAVE form Registration Number (hex): 0x003C Codec ID in the IANA Namespace: audio/vnd.wave;codec=3C WAVE form wFormatTag ID: WAVE_FORMAT_ROCKWELL_DIGITALK WAVEFORMAT Name: Rockwell DIGITALK Contact: Rockwell International Digital Communications Division 4311 Jamboree Rd. PO Box C Newport Beach, California 92658-8902 USA 714-833-4600 

A.41 Xebec 
WAVE form Registration Number (hex): 0x003D Codec ID in the IANA Namespace: audio/vnd.wave;codec= WAVE form wFormatTag ID: WAVE_FORMAT_XEBEC Additonal Information: proprietary compression Contact: David Emberton 44-453-835482 Xebec Multimedia Solutions Limited Smith House 1-3 George Street Nailsworth, Gloucestershire, England GL6 OAG 

A.42 Antex Electronics G.721 
WAVE form Registration Number (hex): 0x0040 Codec ID in the IANA Namespace: audio/vnd.wave;codec=40 WAVE form wFormatTag ID: WAVE_FORMAT_G721_ADPCM Contact: Bob Bauman 310-532-3092 Antex Electronics Coporation 3184-H Airway Ave. Costa Mesa, California 92627 USA 

A.43 G.728 CELP 
WAVE form Registration Number (hex): 0x0041 Codec ID in the IANA Namespace: audio/vnd.wave;codec=41 WAVE form wFormatTag ID: WAVE_FORMAT_G728_CELP Contact: Bob Bauman 310-532-3092 Antex Electronics Coporation 3184-H Airway Ave. Costa Mesa, California 92627 USA 

A.44 MSG723 
WAVE form Registration Number (hex): 0x0042 Codec ID in the IANA Namespace: audio/vnd.wave;codec=42 WAVE form wFormatTag ID: WAVE_FORMAT_MSG723 Contact: Terri Hendry, 425-936-2069 Microsoft Corporation One Microsoft Way Redmond, WA 98052-6399 USA 

A.45 MPEG 
WAVE form Registration Number (hex): 0x0050 Codec ID in the IANA Namespace: audio/vnd.wave;codec=50 WAVE form wFormatTag ID: WAVE_FORMAT_MPEG Contact: Terri Hendry, 425-936-2069 Microsoft Corporation One Microsoft Way Redmond, WA 98052-6399 USA 

A.46 RT24 
WAVE form Registration Number (hex): 0x0052 Codec ID in the IANA Namespace: audio/vnd.wave;codec=52 WAVE form wFormatTag ID: WAVE_FORMAT_RT24 Additional Information: This ACM codec is an alternative codec ID to refer to the Voxware Metavoice codec (Codec ID 0x0074). Only the Voxware reference should be used in the general case. Contact: Alexander V. Sokolsky 717-730-9501 InSoft, Inc. 4718 Old Gettysburg Rd Suite 307 Mechanicsburg, PA 17055-4378 USA 

A.47 PAC
WAVE form Registration Number (hex): 0x0053 Codec ID in the IANA Namespace: audio/vnd.wave;codec=53 WAVE form wFormatTag ID: WAVE_FORMAT_PAC Contact: Alexander V. Sokolsky 717-730-9501 InSoft, Inc. 4718 Old Gettysburg Rd Suite 307 Mechanicsburg, PA 17055-4378 USA

A.48 MPEG Layer 3
WAVE form Registration Number (hex): 0x0055 Codec ID in the IANA Namespace: audio/vnd.wave;codec=55 WAVE form wFormatTag ID: WAVE_FORMAT_MPEGLAYER3 Additional Information: ISO/MPEG Layer3 Format Tag Contact: Tomislav Grcanac (408) 576-1361 AT&T Labs, Inc. 2665 North First Street San Jose, California 95134 USA

A.49 Lucent G.723
WAVE form Registration Number (hex): 0x0059 Codec ID in the IANA Namespace: audio/vnd.wave;codec=59 WAVE form wFormatTag ID: WAVE_FORMAT_LUCENT_G723 Contact: Ray Jones (raykj@lucent.com) Lucent Technologies

A.50 Cirrus
WAVE form Registration Number (hex): 0x0060 Codec ID in the IANA Namespace: audio/vnd.wave;codec=60 WAVE form wFormatTag ID: WAVE_FORMAT_CIRRUS Contact: Mr Scott MacDonald 512 442-7555 Cirrus Logic (USA)

A.51 ESPCM
WAVE form Registration Number (hex): 0x0061 Codec ID in the IANA Namespace: audio/vnd.wave;codec=61 WAVE form wFormatTag ID: WAVE_FORMAT_ESPCM Contact: Paul Sung 510-226-1088 ESS Technology 46107 Landing Parkway Fremont, California 94538 USA

A.52 Voxware
WAVE form Registration Number (hex): 0x0062 Codec ID in the IANA Namespace: audio/vnd.wave;codec=62 WAVE form wFormatTag ID: WAVE_FORMAT_VOXWARE Additional Information: This format is now obsolete Contact: Lee Stewart rover@pipeline.com or compuserve 75570,3525 or lees@voxware.com Voxware Inc 172 Tamarack Circle Skillman, NJ 08558 USA

A.53 Canopus Atrac
WAVE form Registration Number (hex): 0x0063 Codec ID in the IANA Namespace: audio/vnd.wave;codec=63 WAVE form wFormatTag ID: WAVE_FORMAT_CANOPUS_ATRAC Additional Information: ATRACWAVEFORMAT Contact: Masayoshi Araki m-araki@canopus.co.jp 81-78-992-7812 Canopus, Co., Ltd. Kobe Hi-Tech Park 1-2-2 Murotani, Nishi-ku Kobe, Hyogo 651-22 Japan

A.54 G.726 ADPCM
WAVE form Registration Number (hex): 0x0064 Codec ID in the IANA Namespace: audio/vnd.wave;codec=64 WAVE form wFormatTag ID: WAVE_FORMAT_G726_ADPCM Contact: Jean-Claude Anaya 100433.3121@compuserve.com (33) 57-26-99-24 APICOM 218, Avenue du Haut-Leveque Pessac France 33605

A.55 G.722 ADPCM
WAVE form Registration Number (hex): 0x0065 Codec ID in the IANA Namespace: audio/vnd.wave;codec=65 WAVE form wFormatTag ID: WAVE_FORMAT_G722_ADPCM Contact: Jean-Claude Anaya 100433.3121@compuserve.com (33) 57-26-99-24 APICOM 218, Avenue du Haut-Leveque Pessac France 33605

A.56 DSAT
WAVE form Registration Number (hex): 0x0066 Codec ID in the IANA Namespace: audio/vnd.wave;codec=66 WAVE form wFormatTag ID: WAVE_FORMAT_DSAT Contact: Terri Hendry 425-936-2069 Microsoft Corporation One Microsoft Way Redmond, WA 98052-6399 USA

A.57 DSAT Display
WAVE form Registration Number (hex): 0x0067 Codec ID in the IANA Namespace: audio/vnd.wave;codec=67 WAVE form wFormatTag ID: WAVE_FORMAT_DSAT_DISPLAY Contact: Terri Hendry 425-936-2069 Microsoft Corporation One Microsoft Way Redmond, WA 98052-6399 USA

A.58 Voxware Byte Aligned
WAVE form Registration Number (hex): 0x0069 Codec ID in the IANA Namespace: audio/vnd.wave;codec=69 WAVE form wFormatTag ID:n WAVE_FORMAT_VOXWARE_BYTE_ALIGNED Additional Information: This format is now obsolete Contact: Lee Stewart rover@pipeline.com or compuserve 75570,3525 or lees@voxware.com Voxware Inc 172 Tamarack Circle Skillman, NJ 08558 USA

A.59 Voxware AC8
WAVE form Registration Number (hex): 0x0070 Codec ID in the IANA Namespace: audio/vnd.wave;codec=70 WAVE form wFormatTag ID: WAVE_FORMAT_VOXWARE_AC8 Additional Information: This format ID is now obsolete Contact: Lee Stewart lees@voxware.com Voxware Inc. 172 Tamarack Circle Skillman, NJ 08558 USA

A.60 Voxware AC10
WAVE form Registration Number (hex): 0x0071 Codec ID in the IANA Namespace: audio/vnd.wave;codec=71 WAVE form wFormatTag ID: WAVE_FORMAT_VOXWARE_AC10 Additional Information: This format ID is now obsolete Contact: Lee Stewart lees@voxware.com Voxware Inc. 172 Tamarack Circle Skillman, NJ 08558 USA

A.61 Voxware AC16
WAVE form Registration Number (hex): 0x0072 Codec ID in the IANA Namespace: audio/vnd.wave;codec=72 WAVE form wFormatTag ID: WAVE_FORMAT_VOXWARE_AC16 Additional Information: This format ID is now obsolete Contact: Lee Stewart lees@voxware.com Voxware Inc. 172 Tamarack Circle Skillman, NJ 08558 USA

A.62 Voxware AC20
WAVE form Registration Number (hex): 0x0073 Codec ID in the IANA Namespace: audio/vnd.wave;codec=73 WAVE form wFormatTag ID: WAVE_FORMAT_VOXWARE_AC20 Additional Information: This format ID is now obsolete Contact: Lee Stewart lees@voxware.com Voxware Inc. 172 Tamarack Circle Skillman, NJ 08558 USA

A.63 Voxware MetaVoice
WAVE form Registration Number (hex): 0x0074 Codec ID in the IANA Namespace: audio/vnd.wave;codec=74 WAVE form wFormatTag ID: WAVE_FORMAT_VOXWARE_RT24 Additional Information: file and stream oriented Contact: Lee Stewart lees@voxware.com Voxware Inc. 172 Tamarack Circle Skillman, NJ 08558 USA

A.64 Voxware MetaSound
WAVE form Registration Number (hex): 0x0075 Codec ID in the IANA Namespace: audio/vnd.wave;codec=75 WAVE form wFormatTag ID: WAVE_FORMAT_VOXWARE_RT29 Additional Information: file and stream oriented Contact: Lee Stewart lees@voxware.com Voxware Inc. 172 Tamarack Circle Skillman, NJ 08558 USA

A.65 Voxware RT29HW
WAVE form Registration Number (hex): 0x0076 Codec ID in the IANA Namespace: audio/vnd.wave;codec=76 WAVE form wFormatTag ID: WAVE_FORMAT_VOXWARE_RT29HW Additional Information: This format ID is now obsolete Contact: Lee Stewart lees@voxware.com Voxware Inc. 172 Tamarack Circle Skillman, NJ 08558 USA

A.66 Voxware VR12
WAVE form Registration Number (hex): 0x0077 Codec ID in the IANA Namespace: audio/vnd.wave;codec=77 WAVE form wFormatTag ID: WAVE_FORMAT_VOXWARE_VR12 Additional Information: This format ID is now obsolete Contact: Lee Stewart lees@voxware.com Voxware Inc. 172 Tamarack Circle Skillman, NJ 08558 USA

A.67 Voxware VR18
WAVE form Registration Number (hex): 0x0078 Codec ID in the IANA Namespace: audio/vnd.wave;codec=78 WAVE form wFormatTag ID: WAVE_FORMAT_VOXWARE_VR18 Additional Information: This format ID is now obsolete Contact: Lee Stewart lees@voxware.com Voxware Inc. 172 Tamarack Circle Skillman, NJ 08558 USA

A.68 Voxware TQ40
WAVE form Registration Number (hex): 0x0079 Codec ID in the IANA Namespace: audio/vnd.wave;codec=79 WAVE form wFormatTag ID: WAVE_FORMAT_VOXWARE_TQ40 Additional Information: This format ID is now obsolete Contact: Lee Stewart lees@voxware.com Voxware Inc. 172 Tamarack Circle Skillman, NJ 08558 USA

A.69 Softsound
WAVE form Registration Number (hex): 0x0080 Codec ID in the IANA Namespace: audio/vnd.wave;codec=80 WAVE form wFormatTag ID: WAVE_FORMAT_SOFTSOUND Contact: AJ Robinson 44-1727-847949 Softsound, Ltd. 12 St. Stephens Avenue St. Albans, Herts, UK AL3 4AD

A.70 Voxware TQ60
WAVE form Registration Number (hex): 0x0081 Codec ID in the IANA Namespace: audio/vnd.wave;codec=81 WAVE form wFormatTag ID: WAVE_FORMAT_VOXWARE_TQ60 Additional Information: This format ID is now obsolete Contact: Lee Stewart lees@voxware.com Voxware Inc. 172 Tamarack Circle Skillman, NJ 08558 USA

A.71 MSRT24
WAVE form Registration Number (hex): 0x0082 Codec ID in the IANA Namespace: audio/vnd.wave;codec=82 WAVE form wFormatTag ID: WAVE_FORMAT_MSRT24 Additional Information: This ACM codec is an alternative codec ID to refer to the Voxware Metavoice codec (Codec ID 0x0074). Only the Voxware reference should be used in the general case. Contact: Terri Hendry 425-936-2069 Microsoft Corporation One Microsoft Way Redmond, WA 98052-6399 USA

A.72 G.729A
WAVE form Registration Number (hex): 0x0083 Codec ID in the IANA Namespace: audio/vnd.wave;codec=83 WAVE form wFormatTag ID: WAVE_FORMAT_G729A Contact: AT&T Laboratories

A.73 MVI MV12
WAVE form Registration Number (hex): 0x0084 Codec ID in the IANA Namespace: audio/vnd.wave;codec=84 WAVE form wFormatTag ID: WAVE_FORMAT_MVI_MV12 Contact: David R. Whipple whipple@mail.webtek.com Motion Pixels 7802 North 132 East Court Owasso, OK 74055 USA (918) 272-5328

A.74 DF G.726
WAVE form Registration Number (hex): 0x0085 Codec ID in the IANA Namespace: audio/vnd.wave;codec=85 WAVE form wFormatTag ID: WAVE_FORMAT_DF_G726 Contact: Jarno van Rooyen Jarno.VanRooyen@DataVoice.co.za DataFusion Systems (Pty) (Ltd) PO Box 582 Stellenbosch Stellenbosch South Africa 27 21 888 2000

A.75 DF GSM610
WAVE form Registration Number (hex): 0x0086 Codec ID in the IANA Namespace: audio/vnd.wave;codec=86 WAVE form wFormatTag ID: WAVE_FORMAT_DF_GSM610 Contact: Jarno van Rooyen Jarno.VanRooyen@DataVoice.co.za DataFusion Systems (Pty) (Ltd) PO Box 582 Stellenbosch 7600 South Africa 27 21 888 2000

A.76 ISIAudio
WAVE form Registration Number (hex): 0x0088 Codec ID in the IANA Namespace: audio/vnd.wave;codec=88 WAVE form wFormatTag ID: WAVE_FORMAT_ISIAUDIO Contact: Iterated Systems, Inc. 5550-a Peachtree Parkway Suite 650 Norcross, GA 30092 USA 404-840-0633

A.77 Onlive
WAVE form Registration Number (hex): 0x0089 Codec ID in the IANA Namespace: audio/vnd.wave;codec=89 WAVE form wFormatTag ID: WAVE_FORMAT_ONLIVE Contact: Dr. Ajit L. Lalwani ajit@onlive.com (408) 617 - 3595 OnLive! Technologies, Inc. 1039 S. Mary Ave. Sunnyvale, California 94087 USA (408) 617-7000

A.78 SBC24
WAVE form Registration Number (hex): 0x0091 Codec ID in the IANA Namespace: audio/vnd.wave;codec=91 WAVE form wFormatTag ID: WAVE_FORMAT_SBC24 Contact: Dieter Rencken Dieter.W.Rencken@siemenscom.com (408) 492-6539 Siemens Business Communications Systems 4900 Old Ironsides Drive Santa Clara, California 95054 USA (408) 492-2000

A.79 Dolby AC3 SPDIF
WAVE form Registration Number (hex): 0x0092 Codec ID in the IANA Namespace: audio/vnd.wave;codec=92 WAVE form wFormatTag ID: WAVE_FORMAT_DOLBY_AC3_SPDIF Contact: Monty Schmidt Sonic Foundry 100 South Baldwin, Suite 204 Madison, WI 53703 USA 608-256-3133

A.80 ZyXEL ADPCM
WAVE form Registration Number (hex): 0x0097 Codec ID in the IANA Namespace: audio/vnd.wave;codec=97 WAVE form wFormatTag ID: WAVE_FORMAT_ZYXEL_ADPCM Contact: Nasser Tarazi nasser@ZyXEL.COM 714-693-0808 ext 206 ZyXEL Communications, Inc. 4920 E. La Palma Ave Anaheim, California 92807 USA 714-693-0808

A.81 Philips LPCBB
WAVE form Registration Number (hex): 0x0098 Codec ID in the IANA Namespace: audio/vnd.wave;codec=98 WAVE form wFormatTag ID: WAVE_FORMAT_PHILIPS_LPCBB Contact: Kurt Kornmuller Philips Speech Processing Computerstrasse 6 Vienna A-1101 Austria 43 1 601 01

A.82 Packed
WAVE form Registration Number (hex): 0x0099 Codec ID in the IANA Namespace: audio/vnd.wave;codec=99 WAVE form wFormatTag ID: WAVE_FORMAT_PACKED Contact: Alex Ruegg alex.ruegg@studer.ch 41-1-870-1252 Studer Professional Audio AG Althardstrasse 30 Regensdorf, CH 8105 41-1-870-7252

A.83 Rhetorex ADPCM
WAVE form Registration Number (hex): 0x0100 Codec ID in the IANA Namespace: audio/vnd.wave;codec=100 WAVE form wFormatTag ID: WAVE_FORMAT_RHETOREX_ADPCM Contact: Roger Dang roger.dang@octel.com 408-371-0881-x195 Rhetorex, Inc. 200 E Hacienda Ave Campbell, California 95008 USA

A.84 BeCubed Software's IRAT 
WAVE form Registration Number (hex): 0x0101 Codec ID in the IANA Namespace: audio/vnd.wave;codec=101 WAVE form wFormatTag ID: WAVE_FORMAT_IRAT WAVEFORMAT name: Contact: William J. Locke bill@becubed.com BeCubed Software Inc. 1750 Marietta Hwy STE 240 Canton, GA 30114 USA 770-720-1077

A.85 Vivo G.723 
WAVE form Registration Number (hex): 0x00111 Codec ID in the IANA Namespace: audio/vnd.wave;codec=111 WAVE form wFormatTag ID: WAVE_FORMAT_VIVO_G723 Contact: Vivo Software 411 Waverley Oaks Road, Suite 313 Waltham, MA 02154 USA (617) 899-8900 

A.86 Vivo Siren 
WAVE form Registration Number (hex): 0x0112 Codec ID in the IANA Namespace: audio/vnd.wave;codec=112 WAVE form wFormatTag ID: WAVE_FORMAT_VIVO_SIREN Contact: Vivo Software 411 Waverley Oaks Road, Suite 313 Waltham, MA 02154 USA (617) 899-8900 

A.87 Digital G.723
WAVE form Registration Number (hex): 0x0123 Codec ID in the IANA Namespace: audio/vnd.wave;codec=123 WAVE form wFormatTag ID: WAVE_FORMAT_DIGITAL_G723 Contact: John Forecast forecast@shell.lkg.dec.com 508-486-5264 Digital Equipment Corporation 146 Main Street Maynard, MA 01754-2571 USA 1-800-DIGITAL 

A.88 Creative ADPCM 
WAVE form Registration Number (hex): 0x0200 Codec ID in the IANA Namespace: audio/vnd.wave;codec=200 WAVE form wFormatTag ID: WAVE_FORMAT_CREATIVE_ADPCM Contact: Peter Ridge 408-428-2366 Creative Labs, Inc California, USA 

A.89 Creative FastSpeech8 
WAVE form Registration Number (hex): 0x0202 Codec ID in the IANA Namespace: audio/vnd.wave;codec=202 WAVE form wFormatTag ID: WAVE_FORMAT_CREATIVE_FASTSPEECH8 Contact: Peter Ridge 408-428-2366 Creative Labs, Inc California, USA 

A.90 Creative FastSpeech10 
WAVE form Registration Number (hex): 0x0203 Codec ID in the IANA Namespace: audio/vnd.wave;codec=203 WAVE form wFormatTag ID: WAVE_FORMAT_ CREATIVE_FASTSPEECH10 Contact: Peter Ridge 408-428-2366 Creative Labs, Inc California, USA 

A.91 Quarterdeck 
WAVE form Registration Number (hex): 0x0220 Codec ID in the IANA Namespace: audio/vnd.wave;codec=220 WAVE form wFormatTag ID: WAVE_FORMAT_QUARTERDECK Contact: Eugene Olsen 310-309-3700 Quarterdeck Corporation 13160 Mindanao Way FL 3 Marina del Rey, California 90292-9705 USA 

A.92 FM Towns Snd
WAVE form Registration Number (hex): 0x0300 Codec ID in the IANA Namespace: audio/vnd.wave;codec=300 WAVE form wFormatTag ID: WAVE_FORMAT_FM_TOWNS_SND Contact: Fujitsu Corporation

A.93 BTV Digital
WAVE form Registration Number (hex): 0x0400 Codec ID in the IANA Namespace: audio/vnd.wave;codec=400 WAVE form wFormatTag ID: WAVE_FORMAT_BTV_DIGITAL Additional Information: Brooktree digital audio format Contact: Dave Wilson 512-502-1725 Brooktree Corporation 9868 Scranton Road San Diego, California 92121-3707 USA 1-800-228-2777

A.94 VME VMPCM
WAVE form Registration Number (hex): 0x0680 Codec ID in the IANA Namespace: audio/vnd.wave;codec=680 WAVE form wFormatTag ID: WAVE_FORMAT_VME_VMPCM Contact: Tomislav Grcanac (408) 576-1361 AT&T Labs, Inc. 2665 North First Street San Jose, California 95134 USA

A.95 OLIGSM
WAVE form Registration Number (hex): 0x1000 Codec ID in the IANA Namespace: audio/vnd.wave;codec=1000 WAVE form wFormatTag ID: WAVE_FORMAT_OLIGSM Contact: Harry Sinn Ing C. Olivetti & C., S.p.A. Via G. Jervis 77 Via Montalenghe 8 Scarmagno Ivrea (To) 10015 Italy 39-125-527056

A.96 OLIADPCM
WAVE form Registration Number (hex): 0x1001 Codec ID in the IANA Namespace: audio/vnd.wave;codec=1001 WAVE form wFormatTag ID: WAVE_FORMAT_OLIADPCM Contact: Harry Sinn Ing C. Olivetti & C., S.p.A. Via G. Jervis 77 Via Montalenghe 8 Scarmagno Ivrea (To) 10015 Italy 39-125-527056

A.97 OLICELP
WAVE form Registration Number (hex): 0x1002 Codec ID in the IANA Namespace: audio/vnd.wave;codec=1002 WAVE form wFormatTag ID: WAVE_FORMAT_OLICELP Contact: Harry Sinn Ing C. Olivetti & C., S.p.A. Via G. Jervis 77 Via Montalenghe 8 Scarmagno Ivrea (To) 10015 Italy 39-125-527056

A.98 OLISBC
WAVE form Registration Number (hex): 0x1003 Codec ID in the IANA Namespace: audio/vnd.wave;codec=1003 WAVE form wFormatTag ID: WAVE_FORMAT_OLISBC Contact: Harry Sinn Ing C. Olivetti & C., S.p.A. Via G. Jervis 77 Via Montalenghe 8 Scarmagno Ivrea (To) 10015 Italy 39-125-527056

A.99 OLIOPR
WAVE form Registration Number (hex): 0x1004 Codec ID in the IANA Namespace: audio/vnd.wave;codec=1004 WAVE form wFormatTag ID: WAVE_FORMAT_OLIOPR Contact: Harry Sinn Ing C. Olivetti & C., S.p.A. Via G. Jervis 77 Via Montalenghe 8 Scarmagno Ivrea (To) 10015 Italy 39-125-527056

  A.100   LH Codec

WAVE form Registration Number (hex): 0x1100 Codec ID in the IANA Namespace: audio/vnd.wave;codec=1100 WAVE form wFormatTag ID: WAVE_FORMAT_LH_CODEC Contact: David Ray Lernout & Hauspie 20 Mall Road Burlington, MA 01803 USA

  A.101   Norris

WAVE form Registration Number (hex): 0x1400 Codec ID in the IANA Namespace: audio/vnd.wave;codec=1400 WAVE form wFormatTag ID: WAVE_FORMAT_NORRIS Contact: Rick Davis Norris Communications, Inc 12725 Stowe Drive Poway, California 92064 USA 619-679-1504

  A.102   ISIAudio

WAVE form Registration Number (hex): 0x1401 Codec ID in the IANA Namespace: audio/vnd.wave;codec=1401 WAVE form wFormatTag ID: WAVE_FORMAT_ISIAUDIO Contact: Tomislav Grcanac (408) 576-1361 AT&T Labs, Inc. 2665 North First Street San Jose, California 95134 USA

  A.103   Soundspace Music Compression

WAVE form Registration Number (hex): 0x1500 Codec ID in the IANA Namespace: audio/vnd.wave;codec=1500 WAVE form wFormatTag ID: WAVE_FORMAT_SOUNDSPACE_MUSICOMPRESS Contact: Tomislav Grcanac (408) 576-1361 AT&T Labs, Inc. 2665 North First Street San Jose, California 95134 USA

  A.104   DVM

WAVE form Registration Number (hex): 0x2000 Codec ID in the IANA Namespace: audio/vnd.wave;codec=2000 WAVE form wFormatTag ID: WAVE_FORMAT_DVM Contact: Martin Regen FAST Multimedia AG Lansbergerstrasse 76 Munchen 80339 Germany 49-89-50206
}

{
+#define  WAVE_FORMAT_UNKNOWN                    0x0000 /* Microsoft Corporation */
+#define  WAVE_FORMAT_ADPCM                      0x0002 /* Microsoft Corporation */
+#define  WAVE_FORMAT_IEEE_FLOAT                 0x0003 /* Microsoft Corporation */
+#define  WAVE_FORMAT_VSELP                      0x0004 /* Compaq Computer Corp. */
+#define  WAVE_FORMAT_IBM_CVSD                   0x0005 /* IBM Corporation */
+#define  WAVE_FORMAT_ALAW                       0x0006 /* Microsoft Corporation */
+#define  WAVE_FORMAT_MULAW                      0x0007 /* Microsoft Corporation */
+#define  WAVE_FORMAT_DTS                        0x0008 /* Microsoft Corporation */
+#define  WAVE_FORMAT_OKI_ADPCM                  0x0010 /* OKI */
+#define  WAVE_FORMAT_DVI_ADPCM                  0x0011 /* Intel Corporation */
+#define  WAVE_FORMAT_IMA_ADPCM                  (WAVE_FORMAT_DVI_ADPCM) /*  Intel Corporation */
+#define  WAVE_FORMAT_MEDIASPACE_ADPCM           0x0012 /* Videologic */
+#define  WAVE_FORMAT_SIERRA_ADPCM               0x0013 /* Sierra Semiconductor Corp */
+#define  WAVE_FORMAT_G723_ADPCM                 0x0014 /* Antex Electronics Corporation */
+#define  WAVE_FORMAT_DIGISTD                    0x0015 /* DSP Solutions, Inc. */
+#define  WAVE_FORMAT_DIGIFIX                    0x0016 /* DSP Solutions, Inc. */
+#define  WAVE_FORMAT_DIALOGIC_OKI_ADPCM         0x0017 /* Dialogic Corporation */
+#define  WAVE_FORMAT_MEDIAVISION_ADPCM          0x0018 /* Media Vision, Inc. */
+#define  WAVE_FORMAT_CU_CODEC                   0x0019 /* Hewlett-Packard Company */
+#define  WAVE_FORMAT_YAMAHA_ADPCM               0x0020 /* Yamaha Corporation of America */
+#define  WAVE_FORMAT_SONARC                     0x0021 /* Speech Compression */
+#define  WAVE_FORMAT_DSPGROUP_TRUESPEECH        0x0022 /* DSP Group, Inc */
+#define  WAVE_FORMAT_ECHOSC1                    0x0023 /* Echo Speech Corporation */
+#define  WAVE_FORMAT_AUDIOFILE_AF36             0x0024 /* Virtual Music, Inc. */
+#define  WAVE_FORMAT_APTX                       0x0025 /* Audio Processing Technology */
+#define  WAVE_FORMAT_AUDIOFILE_AF10             0x0026 /* Virtual Music, Inc. */
+#define  WAVE_FORMAT_PROSODY_1612               0x0027 /* Aculab plc */
+#define  WAVE_FORMAT_LRC                        0x0028 /* Merging Technologies S.A. */
+#define  WAVE_FORMAT_DOLBY_AC2                  0x0030 /* Dolby Laboratories */
+#define  WAVE_FORMAT_GSM610                     0x0031 /* Microsoft Corporation */
+#define  WAVE_FORMAT_MSNAUDIO                   0x0032 /* Microsoft Corporation */
+#define  WAVE_FORMAT_ANTEX_ADPCME               0x0033 /* Antex Electronics Corporation */
+#define  WAVE_FORMAT_CONTROL_RES_VQLPC          0x0034 /* Control Resources Limited */
+#define  WAVE_FORMAT_DIGIREAL                   0x0035 /* DSP Solutions, Inc. */
+#define  WAVE_FORMAT_DIGIADPCM                  0x0036 /* DSP Solutions, Inc. */
+#define  WAVE_FORMAT_CONTROL_RES_CR10           0x0037 /* Control Resources Limited */
+#define  WAVE_FORMAT_NMS_VBXADPCM               0x0038 /* Natural MicroSystems */
+#define  WAVE_FORMAT_CS_IMAADPCM                0x0039 /* Crystal Semiconductor IMA ADPCM */
+#define  WAVE_FORMAT_ECHOSC3                    0x003A /* Echo Speech Corporation */
+#define  WAVE_FORMAT_ROCKWELL_ADPCM             0x003B /* Rockwell International */
+#define  WAVE_FORMAT_ROCKWELL_DIGITALK          0x003C /* Rockwell International */
+#define  WAVE_FORMAT_XEBEC                      0x003D /* Xebec Multimedia Solutions Limited */
+#define  WAVE_FORMAT_G721_ADPCM                 0x0040 /* Antex Electronics Corporation */
+#define  WAVE_FORMAT_G728_CELP                  0x0041 /* Antex Electronics Corporation */
+#define  WAVE_FORMAT_MSG723                     0x0042 /* Microsoft Corporation */
+#define  WAVE_FORMAT_MPEG                       0x0050 /* Microsoft Corporation */
+#define  WAVE_FORMAT_RT24                       0x0052 /* InSoft, Inc. */
+#define  WAVE_FORMAT_PAC                        0x0053 /* InSoft, Inc. */
+#define  WAVE_FORMAT_MPEGLAYER3                 0x0055 /* ISO/MPEG Layer3 Format Tag */
+#define  WAVE_FORMAT_LUCENT_G723                0x0059 /* Lucent Technologies */
+#define  WAVE_FORMAT_CIRRUS                     0x0060 /* Cirrus Logic */
+#define  WAVE_FORMAT_ESPCM                      0x0061 /* ESS Technology */
+#define  WAVE_FORMAT_VOXWARE                    0x0062 /* Voxware Inc */
+#define  WAVE_FORMAT_CANOPUS_ATRAC              0x0063 /* Canopus, co., Ltd. */
+#define  WAVE_FORMAT_G726_ADPCM                 0x0064 /* APICOM */
+#define  WAVE_FORMAT_G722_ADPCM                 0x0065 /* APICOM */
+#define  WAVE_FORMAT_DSAT_DISPLAY               0x0067 /* Microsoft Corporation */
+#define  WAVE_FORMAT_VOXWARE_BYTE_ALIGNED       0x0069 /* Voxware Inc */
+#define  WAVE_FORMAT_VOXWARE_AC8                0x0070 /* Voxware Inc */
+#define  WAVE_FORMAT_VOXWARE_AC10               0x0071 /* Voxware Inc */
+#define  WAVE_FORMAT_VOXWARE_AC16               0x0072 /* Voxware Inc */
+#define  WAVE_FORMAT_VOXWARE_AC20               0x0073 /* Voxware Inc */
+#define  WAVE_FORMAT_VOXWARE_RT24               0x0074 /* Voxware Inc */
+#define  WAVE_FORMAT_VOXWARE_RT29               0x0075 /* Voxware Inc */
+#define  WAVE_FORMAT_VOXWARE_RT29HW             0x0076 /* Voxware Inc */
+#define  WAVE_FORMAT_VOXWARE_VR12               0x0077 /* Voxware Inc */
+#define  WAVE_FORMAT_VOXWARE_VR18               0x0078 /* Voxware Inc */
+#define  WAVE_FORMAT_VOXWARE_TQ40               0x0079 /* Voxware Inc */
+#define  WAVE_FORMAT_SOFTSOUND                  0x0080 /* Softsound, Ltd. */
+#define  WAVE_FORMAT_VOXWARE_TQ60               0x0081 /* Voxware Inc */
+#define  WAVE_FORMAT_MSRT24                     0x0082 /* Microsoft Corporation */
+#define  WAVE_FORMAT_G729A                      0x0083 /* AT&T Labs, Inc. */
+#define  WAVE_FORMAT_MVI_MVI2                   0x0084 /* Motion Pixels */
+#define  WAVE_FORMAT_DF_G726                    0x0085 /* DataFusion Systems (Pty) (Ltd) */
+#define  WAVE_FORMAT_DF_GSM610                  0x0086 /* DataFusion Systems (Pty) (Ltd) */
+#define  WAVE_FORMAT_ISIAUDIO                   0x0088 /* Iterated Systems, Inc. */
+#define  WAVE_FORMAT_ONLIVE                     0x0089 /* OnLive! Technologies, Inc. */
+#define  WAVE_FORMAT_SBC24                      0x0091 /* Siemens Business Communications Sys */
+#define  WAVE_FORMAT_DOLBY_AC3_SPDIF            0x0092 /* Sonic Foundry */
+#define  WAVE_FORMAT_MEDIASONIC_G723            0x0093 /* MediaSonic */
+#define  WAVE_FORMAT_PROSODY_8KBPS              0x0094 /* Aculab plc */
+#define  WAVE_FORMAT_ZYXEL_ADPCM                0x0097 /* ZyXEL Communications, Inc. */
+#define  WAVE_FORMAT_PHILIPS_LPCBB              0x0098 /* Philips Speech Processing */
+#define  WAVE_FORMAT_PACKED                     0x0099 /* Studer Professional Audio AG */
+#define  WAVE_FORMAT_MALDEN_PHONYTALK           0x00A0 /* Malden Electronics Ltd. */
+#define  WAVE_FORMAT_RHETOREX_ADPCM             0x0100 /* Rhetorex Inc. */
+#define  WAVE_FORMAT_IRAT                       0x0101 /* BeCubed Software Inc. */
+#define  WAVE_FORMAT_VIVO_G723                  0x0111 /* Vivo Software */
+#define  WAVE_FORMAT_VIVO_SIREN                 0x0112 /* Vivo Software */
+#define  WAVE_FORMAT_DIGITAL_G723               0x0123 /* Digital Equipment Corporation */
+#define  WAVE_FORMAT_SANYO_LD_ADPCM             0x0125 /* Sanyo Electric Co., Ltd. */
+#define  WAVE_FORMAT_SIPROLAB_ACEPLNET          0x0130 /* Sipro Lab Telecom Inc. */
+#define  WAVE_FORMAT_SIPROLAB_ACELP4800         0x0131 /* Sipro Lab Telecom Inc. */
+#define  WAVE_FORMAT_SIPROLAB_ACELP8V3          0x0132 /* Sipro Lab Telecom Inc. */
+#define  WAVE_FORMAT_SIPROLAB_G729              0x0133 /* Sipro Lab Telecom Inc. */
+#define  WAVE_FORMAT_SIPROLAB_G729A             0x0134 /* Sipro Lab Telecom Inc. */
+#define  WAVE_FORMAT_SIPROLAB_KELVIN            0x0135 /* Sipro Lab Telecom Inc. */
+#define  WAVE_FORMAT_G726ADPCM                  0x0140 /* Dictaphone Corporation */
+#define  WAVE_FORMAT_QUALCOMM_PUREVOICE         0x0150 /* Qualcomm, Inc. */
+#define  WAVE_FORMAT_QUALCOMM_HALFRATE          0x0151 /* Qualcomm, Inc. */
+#define  WAVE_FORMAT_TUBGSM                     0x0155 /* Ring Zero Systems, Inc. */
+#define  WAVE_FORMAT_MSAUDIO1                   0x0160 /* Microsoft Corporation */
+#define  WAVE_FORMAT_CREATIVE_ADPCM             0x0200 /* Creative Labs, Inc */
+#define  WAVE_FORMAT_CREATIVE_FASTSPEECH8       0x0202 /* Creative Labs, Inc */
+#define  WAVE_FORMAT_CREATIVE_FASTSPEECH10      0x0203 /* Creative Labs, Inc */
+#define  WAVE_FORMAT_UHER_ADPCM                 0x0210 /* UHER informatic GmbH */
+#define  WAVE_FORMAT_QUARTERDECK                0x0220 /* Quarterdeck Corporation */
+#define  WAVE_FORMAT_ILINK_VC                   0x0230 /* I-link Worldwide */
+#define  WAVE_FORMAT_RAW_SPORT                  0x0240 /* Aureal Semiconductor */
+#define  WAVE_FORMAT_IPI_HSX                    0x0250 /* Interactive Products, Inc. */
+#define  WAVE_FORMAT_IPI_RPELP                  0x0251 /* Interactive Products, Inc. */
+#define  WAVE_FORMAT_CS2                        0x0260 /* Consistent Software */
+#define  WAVE_FORMAT_SONY_SCX                   0x0270 /* Sony Corp. */
+#define  WAVE_FORMAT_FM_TOWNS_SND               0x0300 /* Fujitsu Corp. */
+#define  WAVE_FORMAT_BTV_DIGITAL                0x0400 /* Brooktree Corporation */
+#define  WAVE_FORMAT_QDESIGN_MUSIC              0x0450 /* QDesign Corporation */
+#define  WAVE_FORMAT_VME_VMPCM                  0x0680 /* AT&T Labs, Inc. */
+#define  WAVE_FORMAT_TPC                        0x0681 /* AT&T Labs, Inc. */
+#define  WAVE_FORMAT_OLIGSM                     0x1000 /* Ing C. Olivetti & C., S.p.A. */
+#define  WAVE_FORMAT_OLIADPCM                   0x1001 /* Ing C. Olivetti & C., S.p.A. */
+#define  WAVE_FORMAT_OLICELP                    0x1002 /* Ing C. Olivetti & C., S.p.A. */
+#define  WAVE_FORMAT_OLISBC                     0x1003 /* Ing C. Olivetti & C., S.p.A. */
+#define  WAVE_FORMAT_OLIOPR                     0x1004 /* Ing C. Olivetti & C., S.p.A. */
+#define  WAVE_FORMAT_LH_CODEC                   0x1100 /* Lernout & Hauspie */
+#define  WAVE_FORMAT_NORRIS                     0x1400 /* Norris Communications, Inc. */
+#define  WAVE_FORMAT_SOUNDSPACE_MUSICOMPRESS    0x1500 /* AT&T Labs, Inc. */
+#define  WAVE_FORMAT_DVM                        0x2000 /* FAST Multimedia AG */
}
